/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <stdarg.h>
#include <glib/gi18n.h>
#include <eel/eel.h>
#include "gt-conf.h"
#include "gt-util.h"

#define WINDOW_WIDTH_KEY		"gt-conf-window-width-key"
#define WINDOW_HEIGHT_KEY		"gt-conf-window-height-key"

#define RADIO_ACTION_ENUM_TYPE		"gt-conf-radio-action-enum-type"

#define OBJECT_PSPEC_KEY		"gt-conf-object-pspec"

static void gt_conf_link_file_chooser_h (GtkFileChooser *chooser,
					 gpointer user_data);
static void gt_conf_link_file_chooser_notify_cb (GConfClient *client,
						 unsigned int cnxn_id,
						 GConfEntry *entry,
						 gpointer user_data);

static gboolean gt_conf_link_window_h (GtkWidget *widget,
				       GdkEventConfigure *event,
				       gpointer user_data);
static void gt_conf_link_window_notify_cb (GConfClient *client,
					   unsigned int cnxn_id,
					   GConfEntry *entry,
					   gpointer user_data);

static void gt_conf_link_toggle_action_h (GtkToggleAction *action,
					  gpointer user_data);
static void gt_conf_link_toggle_action_notify_cb (GConfClient *client,
						  unsigned int cnxn_id,
						  GConfEntry *entry,
						  gpointer user_data);

static void gt_conf_link_radio_action_set (GtkRadioAction *action,
					   GConfValue *value);
static void gt_conf_link_radio_action_h (GtkRadioAction *action,
					 GtkRadioAction *current,
					 gpointer user_data);
static void gt_conf_link_radio_action_notify_cb (GConfClient *client,
						 unsigned int cnxn_id,
						 GConfEntry *entry,
						 gpointer user_data);

static void gt_conf_link_object_set (GObject *object, GConfValue *value);
static void gt_conf_link_object_h (GObject *object,
				   GParamSpec *pspec,
				   gpointer user_data);
static void gt_conf_link_object_notify_cb (GConfClient *client,
					   unsigned int cnxn_id,
					   GConfEntry *entry,
					   gpointer user_data);

static void gt_conf_notification_add_weak_notify_cb (gpointer data,
						     GObject *former_object);

void
gt_conf_init (void)
{
  /* monitor our namespace */
  eel_gconf_monitor_add(GT_CONF_NAMESPACE);
}

void
gt_conf_link (gpointer object, ...)
{
  va_list args;

  va_start(args, object);
  while (object)
    {
      const char *key;
      char *signal_name;
      GCallback signal_handler;
      GConfClientNotifyFunc notification_cb;

      key = va_arg(args, const char *);
      g_return_if_fail(key != NULL);

      if (GTK_IS_FILE_CHOOSER(object))
	{
	  char *uri;

	  uri = eel_gconf_get_string(key);
	  if (uri)
	    {
	      gtk_file_chooser_set_current_folder_uri(object, uri);
	      g_free(uri);
	    }

	  signal_name = g_strdup("current-folder-changed");
	  signal_handler = G_CALLBACK(gt_conf_link_file_chooser_h);
	  notification_cb = gt_conf_link_file_chooser_notify_cb;
	}
      else if (GTK_IS_WINDOW(object))
	{
	  char *width_key;
	  char *height_key;

	  width_key = g_strdup_printf("%s/width", key);
	  height_key = g_strdup_printf("%s/height", key);

	  g_object_set_data_full(object, WINDOW_WIDTH_KEY, width_key, g_free);
	  g_object_set_data_full(object, WINDOW_HEIGHT_KEY, height_key, g_free);

	  gtk_window_set_default_size(GTK_WINDOW(object),
				      eel_gconf_get_integer(width_key),
				      eel_gconf_get_integer(height_key));

	  signal_name = g_strdup("configure-event");
	  signal_handler = G_CALLBACK(gt_conf_link_window_h);
	  notification_cb = gt_conf_link_window_notify_cb;
	}
      else if (GTK_IS_RADIO_ACTION(object))
	{
	  GType enum_type;
	  GConfValue *value;

	  enum_type = va_arg(args, GType);
	  g_return_if_fail(enum_type != 0);

	  g_object_set_data_full(object, RADIO_ACTION_ENUM_TYPE, g_strdup(g_type_name(enum_type)), g_free);

	  value = eel_gconf_get_value(key);
	  if (value)
	    {
	      gt_conf_link_radio_action_set(object, value);
	      gconf_value_free(value);
	    }

	  signal_name = g_strdup("changed");
	  signal_handler = G_CALLBACK(gt_conf_link_radio_action_h);
	  notification_cb = gt_conf_link_radio_action_notify_cb;
	}
      else if (GTK_IS_TOGGLE_ACTION(object))
	{
	  gtk_toggle_action_set_active(object, eel_gconf_get_boolean(key));
	  
	  signal_name = g_strdup("toggled");
	  signal_handler = G_CALLBACK(gt_conf_link_toggle_action_h);
	  notification_cb = gt_conf_link_toggle_action_notify_cb;
	}
      else if (G_IS_OBJECT(object))
	{
	  const char *property_name;
	  GParamSpec *pspec;
	  GConfValue *value;

	  property_name = va_arg(args, const char *);
	  g_return_if_fail(property_name != NULL);

	  pspec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), property_name);
	  g_return_if_fail(pspec != NULL);

	  g_object_set_data(object, OBJECT_PSPEC_KEY, pspec);

	  value = eel_gconf_get_value(key);
	  if (value)
	    {
	      gt_conf_link_object_set(object, value);
	      gconf_value_free(value);
	    }

	  signal_name = g_strconcat("notify::", property_name, NULL);
	  signal_handler = G_CALLBACK(gt_conf_link_object_h);
	  notification_cb = gt_conf_link_object_notify_cb;
	}
      else
	g_return_if_reached();

      if (signal_name)
	{
	  g_signal_connect_data(object, signal_name, signal_handler, g_strdup(key), (GClosureNotify) g_free, 0);
	  gt_conf_notification_add(object, key, notification_cb, object);
	  g_free(signal_name);
	}

      object = va_arg(args, gpointer);
    }
  va_end(args);
}

static void
gt_conf_link_file_chooser_h (GtkFileChooser *chooser, gpointer user_data)
{
  const char *key = user_data;
  char *uri;

  uri = gtk_file_chooser_get_current_folder_uri(chooser);
  eel_gconf_set_string(key, uri);
  g_free(uri);
}

static void
gt_conf_link_file_chooser_notify_cb (GConfClient *client,
				     unsigned int cnxn_id,
				     GConfEntry *entry,
				     gpointer user_data)
{
  GtkFileChooser *chooser = user_data;
  GConfValue *value = gconf_entry_get_value(entry);
  const char *uri = value ? gconf_value_get_string(value) : NULL;

  if (uri)
    {
      GDK_THREADS_ENTER();
      g_signal_handlers_block_matched(chooser, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_file_chooser_h, NULL);
      gtk_file_chooser_set_current_folder_uri(chooser, uri);
      g_signal_handlers_unblock_matched(chooser, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_file_chooser_h, NULL);
      GDK_THREADS_LEAVE();
    }
}

static gboolean
gt_conf_link_window_h (GtkWidget *widget,
		       GdkEventConfigure *event,
		       gpointer user_data)
{
  const char *width_key = g_object_get_data(G_OBJECT(widget), WINDOW_WIDTH_KEY);
  const char *height_key = g_object_get_data(G_OBJECT(widget), WINDOW_HEIGHT_KEY);

  eel_gconf_set_integer(width_key, event->width);
  eel_gconf_set_integer(height_key, event->height);

  return FALSE;
}

static void
gt_conf_link_window_notify_cb (GConfClient *client,
			       unsigned int cnxn_id,
			       GConfEntry *entry,
			       gpointer user_data)
{
  GtkWindow *window = user_data;
  const char *width_key;
  const char *height_key;

  GDK_THREADS_ENTER();

  width_key = g_object_get_data(G_OBJECT(window), WINDOW_WIDTH_KEY);
  height_key = g_object_get_data(G_OBJECT(window), WINDOW_HEIGHT_KEY);

  g_signal_handlers_block_matched(window, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_window_h, NULL);
  gtk_window_resize(window,
		    eel_gconf_get_integer(width_key),
		    eel_gconf_get_integer(height_key));
  g_signal_handlers_unblock_matched(window, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_window_h, NULL);

  GDK_THREADS_LEAVE();
}

static void
gt_conf_link_radio_action_set (GtkRadioAction *action, GConfValue *value)
{
  const char *enum_type_name;
  GType enum_type;
  GEnumClass *enum_class;
  GEnumValue *enum_value;

  g_return_if_fail(GTK_IS_RADIO_ACTION(action));
  g_return_if_fail(value != NULL);

  enum_type_name = g_object_get_data(G_OBJECT(action), RADIO_ACTION_ENUM_TYPE);
  g_return_if_fail(enum_type_name != NULL);

  enum_type = g_type_from_name(enum_type_name);
  g_return_if_fail(enum_type != 0);

  enum_class = g_type_class_ref(enum_type);
  enum_value = g_enum_get_value_by_nick(enum_class, gconf_value_get_string(value));

  if (enum_value)
    {
      GSList *group;
      GSList *l;

      group = gtk_radio_action_get_group(action);
      GT_LIST_FOREACH(l, group)
        {
	  GtkRadioAction *this_action = l->data;
	  int this_value;

	  g_object_get(this_action, "value", &this_value, NULL);
	  if (this_value == enum_value->value)
	    {
	      gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(this_action), TRUE);
	      break;
	    }
	}
    }
  
  g_type_class_unref(enum_class);
}

static void
gt_conf_link_radio_action_h (GtkRadioAction *action,
			     GtkRadioAction *current,
			     gpointer user_data)
{
  const char *enum_type_name;
  GType enum_type;
  GEnumClass *enum_class;
  int current_value;
  GEnumValue *enum_value;
  const char *key = user_data;

  enum_type_name = g_object_get_data(G_OBJECT(action), RADIO_ACTION_ENUM_TYPE);
  g_return_if_fail(enum_type_name != NULL);

  enum_type = g_type_from_name(enum_type_name);
  g_return_if_fail(enum_type != 0);

  enum_class = g_type_class_ref(enum_type);

  g_object_get(current, "value", &current_value, NULL);

  enum_value = g_enum_get_value(enum_class, current_value);
  g_return_if_fail(enum_value != NULL);

  eel_gconf_set_string(key, enum_value->value_nick);

  g_type_class_unref(enum_class);
}

static void
gt_conf_link_radio_action_notify_cb (GConfClient *client,
				     unsigned int cnxn_id,
				     GConfEntry *entry,
				     gpointer user_data)
{
  GtkRadioAction *action = user_data;
  GConfValue *value = gconf_entry_get_value(entry);

  if (value)
    {
      GDK_THREADS_ENTER();
      g_signal_handlers_block_matched(action, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_radio_action_h, NULL);
      gt_conf_link_radio_action_set(action, value);
      g_signal_handlers_unblock_matched(action, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_radio_action_h, NULL);
      GDK_THREADS_LEAVE();
    }
}

static void
gt_conf_link_toggle_action_h (GtkToggleAction *action,
			      gpointer user_data)
{
  const char *key = user_data;

  eel_gconf_set_boolean(key, gtk_toggle_action_get_active(action));
}

static void
gt_conf_link_toggle_action_notify_cb (GConfClient *client,
				      unsigned int cnxn_id,
				      GConfEntry *entry,
				      gpointer user_data)
{
  GtkToggleAction *action = user_data;
  GConfValue *value = gconf_entry_get_value(entry);
  
  GDK_THREADS_ENTER();
  g_signal_handlers_block_matched(action, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_toggle_action_h, NULL);
  gtk_toggle_action_set_active(action, value ? gconf_value_get_bool(value) : FALSE);
  g_signal_handlers_unblock_matched(action, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_toggle_action_h, NULL);
  GDK_THREADS_LEAVE();
}

static void
gt_conf_link_object_set (GObject *object, GConfValue *value)
{
  GValue gvalue = { 0, };
  GParamSpec *pspec;

  g_return_if_fail(G_IS_OBJECT(object));
  g_return_if_fail(value != NULL);

  pspec = g_object_get_data(object, OBJECT_PSPEC_KEY);
  g_return_if_fail(pspec != NULL);

  g_value_init(&gvalue, G_PARAM_SPEC_VALUE_TYPE(pspec));
  
  if (G_PARAM_SPEC_VALUE_TYPE(pspec) == G_TYPE_BOOLEAN)
    g_value_set_boolean(&gvalue, gconf_value_get_bool(value));
  else if (G_PARAM_SPEC_VALUE_TYPE(pspec) == G_TYPE_INT)
    g_value_set_int(&gvalue, gconf_value_get_int(value));
  else
    g_return_if_reached();
  
  g_object_set_property(object, g_param_spec_get_name(pspec), &gvalue);
  g_value_unset(&gvalue);
}

static void
gt_conf_link_object_h (GObject *object, GParamSpec *pspec, gpointer user_data)
{
  const char *key = user_data;
  GValue value = { 0, };

  g_value_init(&value, G_PARAM_SPEC_VALUE_TYPE(pspec));
  g_object_get_property(object, g_param_spec_get_name(pspec), &value);

  if (G_PARAM_SPEC_VALUE_TYPE(pspec) == G_TYPE_BOOLEAN)
    eel_gconf_set_boolean(key, g_value_get_boolean(&value));
  else if (G_PARAM_SPEC_VALUE_TYPE(pspec) == G_TYPE_INT)
    eel_gconf_set_integer(key, g_value_get_int(&value));
  else
    g_return_if_reached();

  g_value_unset(&value);
}

static void
gt_conf_link_object_notify_cb (GConfClient *client,
			       unsigned int cnxn_id,
			       GConfEntry *entry,
			       gpointer user_data)
{
  GObject *object = user_data;
  GConfValue *value = gconf_entry_get_value(entry);

  if (value)
    {
      GDK_THREADS_ENTER();
      g_signal_handlers_block_matched(object, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_object_h, NULL);
      gt_conf_link_object_set(object, value);
      g_signal_handlers_unblock_matched(object, G_SIGNAL_MATCH_FUNC, 0, 0, NULL, gt_conf_link_object_h, NULL);
      GDK_THREADS_LEAVE();
    }
}

void
gt_conf_notification_add (gpointer object,
			  const char *key,
			  GConfClientNotifyFunc callback,
			  gpointer user_data)
{
  unsigned int notification_id;

  g_return_if_fail(G_IS_OBJECT(object));
  g_return_if_fail(key != NULL);
  g_return_if_fail(callback != NULL);

  notification_id = eel_gconf_notification_add(key, callback, user_data);
  g_object_weak_ref(G_OBJECT(object), gt_conf_notification_add_weak_notify_cb, GUINT_TO_POINTER(notification_id));
}

static void
gt_conf_notification_add_weak_notify_cb (gpointer data, GObject *former_object)
{
  unsigned int notification_id = GPOINTER_TO_UINT(data);
  eel_gconf_notification_remove(notification_id);
}

GSList *
gt_conf_get_services (void)
{
  GSList *l;
  GSList *conf_services;
  GSList *services = NULL;

  conf_services = eel_gconf_get_string_list(GT_CONF_SERVICES);

  GT_LIST_FOREACH(l, conf_services)
    {
      const char *entry = l->data;
      char *s;
      char *name;
      gboolean enabled;
      TranslateService *service;
      
      s = strrchr(entry, ':');
      if (! s)
	{
	  g_warning(_("in configuration key \"%s\": invalid entry \"%s\", ignoring"), GT_CONF_SERVICES, entry);
	  continue;
	}

      if (! strcmp(s + 1, "enabled"))
	enabled = TRUE;
      else if (! strcmp(s + 1, "disabled"))
	enabled = FALSE;
      else
	{
	  g_warning(_("in configuration key \"%s\": invalid keyword \"%s\", ignoring"), GT_CONF_SERVICES, s + 1);
	  continue;
	}

      name = g_strndup(entry, s - entry);
      service = translate_get_service(name);
      if (service)
	{
	  services = g_slist_append(services, gt_conf_service_new(service, enabled));
	  g_object_unref(service);
	}
      else
	g_warning(_("in configuration key \"%s\": unknown service \"%s\", ignoring"), GT_CONF_SERVICES, name);
      g_free(name);
    }

  eel_g_slist_free_deep(conf_services);

  return services;
}

void
gt_conf_set_services (const GSList *services)
{
  GSList *conf_services = NULL;
  const GSList *l;

  GT_LIST_FOREACH(l, services)
    {
      const GTConfService *service = l->data;
      char *entry;

      entry = g_strdup_printf("%s:%s", translate_service_get_name(service->service), service->enabled ? "enabled" : "disabled");
      conf_services = g_slist_append(conf_services, entry);
    }

  eel_gconf_set_string_list(GT_CONF_SERVICES, conf_services);
  eel_g_slist_free_deep(conf_services);
}

GTConfService *
gt_conf_service_new (TranslateService *service, gboolean enabled)
{
  GTConfService *conf_service;

  g_return_val_if_fail(TRANSLATE_IS_SERVICE(service), NULL);

  conf_service = g_new0(GTConfService, 1);
  conf_service->service = g_object_ref(service);
  conf_service->enabled = enabled;

  return conf_service;
}

void
gt_conf_service_free (GTConfService *service)
{
  g_return_if_fail(service != NULL);

  g_object_unref(service->service);
  g_free(service);
}

void
gt_conf_services_free (GSList *services)
{
  eel_g_slist_free_deep_custom(services, (GFunc) gt_conf_service_free, NULL);
}

const GTConfService *
gt_conf_services_get_from_name (const GSList *services, const char *name)
{
  const GSList *l;

  GT_LIST_FOREACH(l, services)
    {
      const GTConfService *service = l->data;

      if (! strcmp(translate_service_get_name(service->service), name))
	return service;
    }
  
  return NULL;
}
