-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-statement.adb,v $
--  Description     : Ada Database Objects - Statement Object                --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/23 22:04:21 $
--  Version         : $Revision: 1.3 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2001 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  The statement object contains all informations about argument and        --
--  parsed chunks of a statement.                                            --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  Only Linux                                                               --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
--* ADA
with Ada.Strings.Unbounded;                    use Ada.Strings.Unbounded;
with Ada.Exceptions;                           use Ada.Exceptions;
with Ada.Tags;                                 use Ada.Tags;
with Ada.Text_IO;                              use Ada.Text_IO;
with Unchecked_Deallocation;

--* ADO
with GNU.DB.ADBC.Driver;                       use GNU.DB.ADBC.Driver;
with GNU.DB.ADBC.Hostvariable;                 use GNU.DB.ADBC.Hostvariable;
with GNU.DB.ADBC.Hostvariable.Types;           use GNU.DB.ADBC.Hostvariable.Types;
with GNU.DB.ADBC.Connection;                   use GNU.DB.ADBC.Connection;
with GNU.DB.ADBC.Resultset;                    use GNU.DB.ADBC.Resultset;

package body GNU.DB.ADBC.Statement is

   Version : constant String :=
      "$Id: gnu-db-adbc-statement.adb,v 1.3 2002/02/23 22:04:21 merdmann Exp $";

   Host_Variable_Prefix : constant Character := ':';

   ---------------------
   -- Statement Table --
   ---------------------
   subtype ST_Index is Statement_ID range Null_ID+1..Statement_ID'Last;

   ST : array( ST_Index ) of Handle := (others => Null_Handle );

   ---====================================================================---
   ---===                O B J  E C T     D A T A                      ===---
   ---===                                                              ===---
   ---=== This section contains all declarations of data structures    ===---
   ---=== to implement one instance of the connection object           ===---
   ---===                                                              ===---
   ---====================================================================---

   type Parser_Index is new Positive range 1..256;

   ------------------
   -- Parse_record --
   ------------------
   type Parse_Record is record
         Cmd : Unbounded_String := Null_Unbounded_String;
         Arg : Unbounded_String := Null_Unbounded_String;
      end record;

   type Parse_Table is array( Parser_Index ) of Parse_Record;

   -----------------
   -- Bind_Record --
   -----------------
   type Bind_Record is record
         Name : Unbounded_String    := Null_Unbounded_String;
         Var  : Hostvariable.Handle := null;
      end record;

   type Bind_Table is array( 1..50 ) of Bind_Record;

   type Result_Table is array( Resultset_ID ) of Boolean;
   -----------------
   -- Object_Data --
   -----------------
   type Object_Data is record
         PT    : Parse_Table;
         BT    : Bind_Table;
         Query : Unbounded_String := Null_Unbounded_String;
         RT    : Result_Table;
      end record;

   ---=====================================================================---
   ---===         L O C A L   S U P P O R T   P R O C E D U R E S       ===---
   ---===                                                               ===---
   ---=====================================================================---

   ----------------
   -- Copy_Query --
   ----------------
   procedure Copy_Query(
     S      : in String;
     Next   : in out Positive;
     Result : out String;
     Length : out Positive ) is
     -- copy parts of the string till the begin of a host variable
     J      : Positive := Result'First;
   begin

     for I in S'Range loop
        Next := I;
        exit when S(Next) = Host_Variable_Prefix;
        Result(J) := S(I);
        Length    := J;

        J := J + 1;
        exit when not ( J in Result'Range );
     end loop;
   end Copy_Query;

   -----------------------
   -- Copy_Hostvariable --
   -----------------------
   procedure Copy_Hostvariable(
     S      : in String;
     Next   : in out Positive;
     Result : out String;
     Length : out Positive ) is
     -- copy parts of the string till the begin of a host variable
     J      : Positive := Result'First;
   begin
     for I in S'Range loop
        Next := I;
        exit when S(Next) = ' ';

        Result(J) := S(I);
        Length    := J;

        J := J + 1;
        exit when not ( J in Result'Range );
     end loop;
   end Copy_Hostvariable;

   -----------
   -- Parse --
   -----------
   procedure Parse(
      P    : in out Parse_Table;
      Stmt : in String ) is
      -- prepare a statement, which means that the SQL statement is broken up
      -- into fixed strings and variable parts.
      Next   : Positive := Stmt'First;
      Length : Natural  := 0;
      Result : String( 1..1024 );
   begin
      for I in P'Range loop
         Copy_Query( Stmt(Next..Stmt'Last), Next, Result, Length );
         P(I).Cmd := To_Unbounded_String( Result( 1..Length) );

         if Stmt(Next) = Host_Variable_Prefix then
            Next := Next + 1;
            Copy_Hostvariable( Stmt(Next..Stmt'Last), Next, Result, Length );
            P(I).Arg := To_Unbounded_String( Result( 1..Length) );
            exit when Next = Stmt'Last;
         else
            exit;
         end if;
      end loop;

   end Parse;

   --------------------
   -- Variable_Useed --
   --------------------
   function Variable_Used(
      P    : in Parse_Table;
      Name : in String ) return Boolean is
   begin
      for I in P'Range loop
         if P(I).Arg /= Null_Unbounded_String and then P(I).Arg = Name then
            return True;
         end if;
      end loop;

      return False;
   end Variable_Used;

   ----------------------
   -- Create_Exception --
   ----------------------
   procedure Create_Exception(
      Stmt : in Statement_ID;
      Id   : in Exception_Id;
      -- create an exception and provide some additional information
      -- which might be interesting for debugging.
      Info : in String )  is

      Data : Object_Data_Access := ST(Stmt).Data;
      Q    : Unbounded_String;
   begin
      for I in Data.PT'Range loop
         exit when Data.PT(I).Cmd = Null_Unbounded_String;
         Q := Q & Data.PT(I).Cmd;
         if Data.Pt(I).Arg /= Null_Unbounded_String then
            Q := Q & ":" & Data.Pt(I).Arg;
         end if;
      end loop;
      Raise_Exception( Id, Info & " in " & To_String(Q) );
   end Create_Exception;

   ------------
   -- Parent --
   ------------
   function Parent(
      Stmt : in Statement_ID ) return Driver.Handle is
      -- derive the parent driver from the statement id.
   begin
      if ST(Stmt) /= null then
         return Connection.Driver_Handle( ST(Stmt).Con ) ;
      else
         return null;
      end if;
   end Parent;

   --------------------
   -- Drop_Resources --
   --------------------
   procedure Drop_Resources(
      Stmt : in Statement_ID ) is
      -- drop all resources from the connection
      Data : Object_Data_Access renames ST(Stmt).Data;
   begin
      for I in Data.RT'Range loop
         if Data.RT(I) = True then
            if St(Stmt) /= null then
               Resultset.Deallocate( I );
            end if;
            Data.RT(I) := False;
         end if;
      end loop;
   end Drop_Resources;

   ---======================================================================---
   ---===             C O M P O  N E N T    I N T E R F A C E            ===---
   ---======================================================================---

   ---=====================================================================---
   ---===           A T T R I B U T E    F U N C T I O N S              ===---
   ---=====================================================================---

   ----------------
   -- Get_Handle --
   ----------------
   function Get_Handle(
      Stmt   : in Statement_ID ) return Handle is
   begin
      return ST(Stmt);
   end Get_Handle;

   -------------------
   -- Driver_Handle --
   -------------------
   function Driver_Handle(
      Stmt   : in Statement_ID ) return Driver.Handle is
   begin
      return Parent( Stmt );
   end Driver_Handle;


   ---=====================================================================---
   ---===                        M E T H O D S                          ===---
   ---=====================================================================---

   --------------
   -- Allocate --
   --------------
   function Allocate(
      Stmt : in Handle ) return Statement_ID is
      -- store a handle in the statement table. This will be called by the
      -- driver to insert.
   begin
      if Stmt = Null_Handle then
         raise Usage_Error;
      end if;

      for I in ST'Range loop
         if ST(I) = Null_Handle then
            ST(I) := stmt;
            if Stmt.Data = null then
               Stmt.Data := new Object_Data;
            end if;
            Add_Statement( Stmt.Con, I );
            return I;
         end if;
      end loop;

      raise Statement_Table_Overflow;
   end Allocate;

   ----------------
   -- Deallocate --
   ----------------
   procedure Deallocate(
      Id  : in Statement_ID ) is
      -- deallocate the id from the resulset table
      Db  : Driver.Handle renames Parent(Id);
      procedure Free is
            new Unchecked_Deallocation( Object_Data, Object_Data_Access);
   begin
      if ST(Id) /= null then
         if ST(Id).Data /= null then
            Drop_Resources( Id );
            Delete_Statement( ST(Id).Con, Id );
            Free( ST(Id).Data );

            Delete_Statement( Db.all, Id );
         end if;
      end if;
      ST(Id) := null;
   end Deallocate;

   -------------------
   -- Add_Resultset --
   -------------------
   procedure Add_Resultset(
      Stmt   : in Statement_ID;
      Result : in Resultset_ID ) is
      Data   : Object_Data_Access renames ST(Stmt).Data;
   begin
      Data.RT( Result ) := True;
   end Add_Resultset;

   ----------------------
   -- Delete_Resultset --
   ----------------------
   procedure Delete_Resultset(
      Stmt   : in Statement_ID;
      Result : in Resultset_ID ) is
      -- remove a resultset from a statement
   begin
      pragma Assert( Stmt in ST'Range );

      if ST(Stmt) /= null then
         ST(Stmt).Data.RT( Result ) := False;
      end if;
   end Delete_Resultset;

   --------------
   -- Prepare  --
   --------------
   procedure Prepare(
      Stmt      : in Statement_ID;
      Statement : in String ) is
      S         : Handle renames ST(Stmt);
   begin
      for I in Parser_Index loop
         S.Data.PT(I).Cmd := Null_Unbounded_String;
         S.Data.PT(I).Arg := Null_Unbounded_String;
      end loop;

      Parse( S.Data.PT, Statement );
   end Prepare;

   ----------
   -- Bind --
   ----------
   procedure Bind(
      Stmt : in Statement_ID;
      Name : in String;
      Var  : Hostvariable.Object'Class ) is
      -- bind a hostvariable name to a variable
      S    : Handle renames ST(Stmt);
   begin
      if not Variable_Used( S.Data.PT, Name ) then
         Create_Exception( Stmt, Unused_Host_Variable'Identity, Name );
      end if;

      for I in S.Data.BT'Range loop
         if S.Data.BT(I).Var = null then
            S.Data.BT(I).Var  := Self( Var );
            S.Data.BT(I).Name := To_Unbounded_String(Name);
            return;
         end if;
      end loop;

      raise Bind_Table_Overflow;
   end Bind;

   --------------
   -- Variable --
   --------------
   function Variable(
      Stmt : in Statement_ID;
      Name : in String ) return Hostvariable.Handle is
      -- return the handle
      S    : Handle renames ST(Stmt);
   begin
      for I in S.Data.BT'Range loop
         if To_String(S.Data.BT(I).Name) = Name then
            return S.Data.BT(I).Var;
         end if;
      end loop;

      Create_Exception( Stmt, Host_Variable_Not_Found'Identity, Name );
      return null;
   end Variable;

   ----------
   -- Bind --
   ----------
   procedure Bind_Host_Variables(
      Stmt : in Statement_ID ) is
      -- bind a hostvariables of a statement to the database interface
      Db   : Driver.Handle renames Parent( Stmt );
      Data : Object_Data_Access renames ST(Stmt).Data ;
   begin
      for I in Data.BT'Range loop
         if Data.BT(I).Var /= null then
            Bind_Host_Variable( Db.all, Stmt, Data.BT(I).Var );
         end if;
      end loop;
   end Bind_Host_Variables;

   ----------------------------
   -- Retrieve_Host_Variable --
   ----------------------------
   procedure Retrive_Host_Variables(
      Stmt   : in Statement_ID;
      Result : in Row.Handle ) is
      -- copy the contents of the bound hostvariables into the ada95
      -- presentation.
      Db   : Driver.Handle renames Parent( Stmt );
      Data : Object_Data_Access renames ST(Stmt).Data ;
   begin
      for I in Data.BT'Range loop
         if Data.BT(I).Var /= null then
            Get_Host_Value( Db.all, Stmt, Data.BT(I).Var );
         end if;
      end loop;
   end Retrive_Host_Variables;

   -----------
   -- Query --
   -----------
   function Query(
      Stmt   : in Statement_ID) return String is
      --
      S      : Handle renames ST(Stmt);
      PT     : Parse_Table renames S.Data.PT;
      Q      : Unbounded_String := Null_Unbounded_String;

      function Variable(
         Name : in Unbounded_String ) return Hostvariable.Handle is
      begin
         for I in S.Data.BT'Range loop
            if S.Data.BT(I).Name = Name then
               return S.Data.BT(I).Var;
            end if;
         end loop;
         Create_Exception(
            Stmt,
            Host_Variable_Not_Found'Identity,
            To_String(Name)
         );
         return null;
      end Variable;

      V      : Hostvariable.Handle := null;
      Db     : Driver.Handle renames Parent(Stmt) ;
   begin
      for I in Parser_Index'Range loop
         exit when PT(I).Cmd = Null_Unbounded_String;
         Q := Q & PT(I).Cmd ;
         if Pt(I).Arg /= Null_Unbounded_String then
            V := Variable(PT(I).Arg);
            Q := Q & Expand( Db.all, V);
         end if;

      end loop;

      return To_String(Q);
   end Query;

end GNU.DB.ADBC.Statement;

